# SPDX-License-Identifier: AGPL-3.0-or-later
"""
FreedomBox app for monkeysphere.
"""

from django.utils.translation import gettext_lazy as _

from plinth import app as app_module
from plinth import menu
from plinth.modules.backups.components import BackupRestore
from plinth.modules.users.components import UsersAndGroups
from plinth.package import Packages

from . import manifest

_description = [
    _('With Monkeysphere, an OpenPGP key can be generated for each configured '
      'domain serving SSH. The OpenPGP public key can then be uploaded to the '
      'OpenPGP keyservers. Users connecting to this machine through SSH can '
      'verify that they are connecting to the correct host.  For users to '
      'trust the key, at least one person (usually the machine owner) must '
      'sign the key using the regular OpenPGP key signing process.  See the '
      '<a href="http://web.monkeysphere.info/getting-started-ssh/"> '
      'Monkeysphere SSH documentation</a> for more details.'),
    _('Monkeysphere can also generate an OpenPGP key for each Secure Web '
      'Server (HTTPS) certificate installed on this machine. The OpenPGP '
      'public key can then be uploaded to the OpenPGP keyservers. Users '
      'accessing the web server through HTTPS can verify that they are '
      'connecting to the correct host. To validate the certificate, the user '
      'will need to install some software that is available on the '
      '<a href="https://web.monkeysphere.info/download/"> Monkeysphere '
      'website</a>.')
]

app = None


class MonkeysphereApp(app_module.App):
    """FreedomBox app for Monkeysphere."""

    app_id = 'monkeysphere'

    _version = 1

    def __init__(self):
        """Create components for the app."""
        super().__init__()

        info = app_module.Info(app_id=self.app_id, version=self._version,
                               name=_('Monkeysphere'), icon='fa-certificate',
                               description=_description,
                               manual_page='Monkeysphere')
        self.add(info)

        menu_item = menu.Menu('menu-monkeysphere', info.name, None, info.icon,
                              'monkeysphere:index', parent_url_name='system',
                              advanced=True)
        self.add(menu_item)

        packages = Packages('packages-monkeysphere', ['monkeysphere'])
        self.add(packages)

        users_and_groups = UsersAndGroups('users-and-groups-monkeysphere',
                                          reserved_usernames=['monkeysphere'])
        self.add(users_and_groups)

        backup_restore = BackupRestore('backup-restore-monkeysphere',
                                       **manifest.backup)
        self.add(backup_restore)


def setup(helper, old_version=None):
    """Install and configure the module."""
    app.setup(old_version)
