/*
 * Copyright 2017 Google Inc.
 *
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

#ifndef GrMtlUtil_DEFINED
#define GrMtlUtil_DEFINED

#import <Metal/Metal.h>

#include "include/gpu/GrBackendSurface.h"
#include "include/private/GrTypesPriv.h"
#include "src/sksl/ir/SkSLProgram.h"

#if defined(SK_BUILD_FOR_MAC)
#if __MAC_OS_X_VERSION_MAX_ALLOWED < 101400
#error Must use at least 10.14 SDK to build Metal backend for MacOS
#endif
#else
#if __IPHONE_OS_VERSION_MAX_ALLOWED < 120000 && __TV_OS_VERSION_MAX_ALLOWED < 120000
#error Must use at least 12.00 SDK to build Metal backend for iOS
#endif
#endif

class GrMtlGpu;
class GrSurface;

/**
 * Returns a id<MTLTexture> to the MTLTexture pointed at by the const void*.
 */
SK_ALWAYS_INLINE id<MTLTexture> GrGetMTLTexture(const void* mtlTexture)  {
#if __has_feature(objc_arc)
    return (__bridge id<MTLTexture>)mtlTexture;
#else
    // ARC will retain when bridging from a CoreFoundation to an ObjC object
    return (id<MTLTexture>) CFRetain(mtlTexture);
#endif
}

/**
 * Returns a const void* to whatever the id object is pointing to.
 */
SK_ALWAYS_INLINE const void* GrGetPtrFromId(id idObject) {
#if __has_feature(objc_arc)
    return (__bridge const void*)idObject;
#else
    return (const void*)idObject;
#endif
}

/**
 * Returns a const void* to whatever the id object is pointing to.
 * Will call CFRetain on the object.
 */
SK_ALWAYS_INLINE const void* GrRetainPtrFromId(id idObject) {
    return CFBridgingRetain(idObject);
}

enum class GrMtlErrorCode {
    kTimeout = 1,
};

NSError* GrCreateMtlError(NSString* description, GrMtlErrorCode errorCode);

/**
 * Returns a MTLTextureDescriptor which describes the MTLTexture. Useful when creating a duplicate
 * MTLTexture without the same storage allocation.
 */
MTLTextureDescriptor* GrGetMTLTextureDescriptor(id<MTLTexture> mtlTexture);

/**
 * Returns a compiled MTLLibrary created from MSL code generated by SkSLC
 */
id<MTLLibrary> GrGenerateMtlShaderLibrary(const GrMtlGpu* gpu,
                                          const SkSL::String& sksl,
                                          SkSL::ProgramKind kind,
                                          const SkSL::Program::Settings& settings,
                                          SkSL::String* msl,
                                          SkSL::Program::Inputs* outInputs,
                                          GrContextOptions::ShaderErrorHandler* errorHandler);

/**
* Returns a compiled MTLLibrary created from MSL code
*/
id<MTLLibrary> GrCompileMtlShaderLibrary(const GrMtlGpu* gpu,
                                         const SkSL::String& msl,
                                         GrContextOptions::ShaderErrorHandler* errorHandler);

/**
 * Replacement for newLibraryWithSource:options:error that has a timeout.
 */
id<MTLLibrary> GrMtlNewLibraryWithSource(id<MTLDevice>, NSString* mslCode,
                                         MTLCompileOptions*, NSError**);

/**
 * Replacement for newRenderPipelineStateWithDescriptor:error that has a timeout.
 */
id<MTLRenderPipelineState> GrMtlNewRenderPipelineStateWithDescriptor(
        id<MTLDevice>, MTLRenderPipelineDescriptor*, NSError**);

/**
 * Returns a MTLTexture corresponding to the GrSurface.
 */
id<MTLTexture> GrGetMTLTextureFromSurface(GrSurface* surface);

static inline MTLPixelFormat GrBackendFormatAsMTLPixelFormat(const GrBackendFormat& format) {
    return static_cast<MTLPixelFormat>(format.asMtlFormat());
}

/**
 * Returns true if the format is compressed.
 */
bool GrMtlFormatIsCompressed(MTLPixelFormat mtlFormat);

/**
 * Maps a MTLPixelFormat into the CompressionType enum if applicable.
 */
SkImage::CompressionType GrMtlFormatToCompressionType(MTLPixelFormat mtlFormat);

size_t GrMtlFormatBytesPerBlock(MTLPixelFormat);

int GrMtlFormatStencilBits(MTLPixelFormat);

#endif
