/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <QAction>
#include <QDebug>
#include <QFile>
#include <QFileDialog>
#include <QMenu>
#include <QPlainTextEdit>
#include <QPushButton>
#include <QSettings>
#include <QSplitter>
#include <QStatusBar>
#include <QTreeWidget>
#include <QTreeWidgetItem>
#include <QVBoxLayout>

#include "FlextraExaminer.h"
#include "LogHandler.h"

#include "mars.h"

#include "MvFlextra.h"
#include "MvQAbout.h"
#include "MvQFileInfoLabel.h"
#include "MvQHighlighter.h"
#include "MvQLogBrowser.h"
#include "MvQPlainTextEditSearchLine.h"

FlextraExaminer::FlextraExaminer(QWidget *parent) :
		MvQMainWindow(parent),
		data_(0),actionFileInfo_(0), actionLog_(0)
	
{
	// Initializations
 	setAttribute(Qt::WA_DeleteOnClose);	
	setWindowTitle(tr("Metview - FLEXTRA Examiner"));

	//Initial size
	setInitialSize(1100,800);

	setupBlockPanel();
	setupDumpPanel();
	setupLogArea();

	// Main splitter
	mainSplitter_= new QSplitter;
	mainSplitter_->setOrientation(Qt::Vertical);
	mainSplitter_->setOpaqueResize(false);
	setCentralWidget(mainSplitter_);
	
	// The main layout (the upper part of mainSplitter)
	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->setObjectName(QString::fromUtf8("vboxLayout"));
	mainLayout->setContentsMargins(0,0,0,0);
	mainLayout->setSpacing(1);
	QWidget* w = new QWidget;
	w->setLayout(mainLayout);
	mainSplitter_->addWidget(w);

	// File info area
	fileInfoLabel_= new MvQFileInfoLabel;
	mainLayout->addWidget(fileInfoLabel_);
		
	// Central splitter
	centralSplitter_= new QSplitter;
	centralSplitter_->setOrientation(Qt::Horizontal);
	centralSplitter_->setOpaqueResize(false);
	mainLayout->addWidget(centralSplitter_);
	
	centralSplitter_->addWidget(blockPanel_);
	centralSplitter_->addWidget(dumpPanel_);
	centralSplitter_->setCollapsible(1,false);
	QList<int> splitterSize;
	splitterSize << 200 << 900;
	centralSplitter_->setSizes(splitterSize);
	
	//Log
	mainSplitter_->addWidget(logPanel_);

	// Initialize user interface actions
	setupEditActions();
	setupViewActions();
	setupHelpActions();

	//----------------------------
	// Setup menus and toolbars
	//----------------------------

	setupMenus(menuItems_);
	
	// Status bar
 	statusMessageLabel_ = new QLabel("");
    	statusMessageLabel_->setFrameShape(QFrame::NoFrame);
    	statusBar()->addPermanentWidget(statusMessageLabel_,1);  // '1' means 'please stretch me when resized'

	// Signals and slots
	connect(this,SIGNAL(statusMessage(QString)),
		this,SLOT(slotStatusMessage(QString)));

	// Setup previous main layout
	readSettings();
}
 
FlextraExaminer::~FlextraExaminer()
{
	// Save settings
	writeSettings();

	// Clean memory
	if(data_) delete data_;
}


void FlextraExaminer::setupEditActions()
{
	QAction *actionFind_=MvQMainWindow::createAction(MvQMainWindow::FindAction,this);
	connect(actionFind_,SIGNAL(triggered(bool)),
		findPanel_,SLOT(setHidden(bool)));
				
	QAction *actionFindNext=MvQMainWindow::createAction(MvQMainWindow::FindNextAction,this);
	connect(actionFindNext,SIGNAL(triggered(bool)),
		findPanel_,SLOT(slotFindNext(bool)));	
		
	QAction *actionFindPrev=MvQMainWindow::createAction(MvQMainWindow::FindPreviousAction,this);
	connect(actionFindPrev,SIGNAL(triggered(bool)),
		findPanel_,SLOT(slotFindPrev(bool)));
		
	MvQMainWindow::MenuType menuType=MvQMainWindow::EditMenu;
	
	menuItems_[menuType].push_back(new MvQMenuItem(actionFind_,MvQMenuItem::MenuTarget));
	menuItems_[menuType].push_back(new MvQMenuItem(actionFindNext,MvQMenuItem::MenuTarget));
	menuItems_[menuType].push_back(new MvQMenuItem(actionFindPrev,MvQMenuItem::MenuTarget));
}

void FlextraExaminer::setupViewActions()
{
	actionFileInfo_ = new QAction(this);
	actionFileInfo_->setObjectName(QString::fromUtf8("actionFileInfo"));
	actionFileInfo_->setText(tr("File info"));
	actionFileInfo_->setCheckable(true);
	actionFileInfo_->setChecked(true);
	actionFileInfo_->setToolTip(tr("View file info"));
	QIcon icon;
	icon.addPixmap(QPixmap(QString::fromUtf8(":/examiner/fileInfo.svg")), QIcon::Normal, QIcon::Off);
 	actionFileInfo_->setIcon(icon);	

	actionLog_=MvQMainWindow::createAction(MvQMainWindow::LogAction,this);
	
	logPanel_->hide(); // hide log area
	
	// Signals and slots
	connect(actionFileInfo_,SIGNAL(toggled(bool)),
		fileInfoLabel_,SLOT(setVisible(bool)));

	connect(actionLog_,SIGNAL(toggled(bool)),
		logPanel_,SLOT(setVisible(bool)));

	MvQMainWindow::MenuType menuType=MvQMainWindow::ViewMenu;
	menuItems_[menuType].push_back(new MvQMenuItem(actionFileInfo_));	
	menuItems_[menuType].push_back(new MvQMenuItem(actionLog_));
}

void FlextraExaminer::setupHelpActions()
{
    	// About
    	QAction *actionAbout=MvQMainWindow::createAction(MvQMainWindow::AboutAction,this);
	actionAbout->setText(tr("&About FLEXTRA Examiner"));
	connect(actionAbout, SIGNAL(triggered()), this, SLOT(slotShowAboutBox()));

	MvQMainWindow::MenuType menuType=MvQMainWindow::HelpMenu;
	menuItems_[menuType].push_back(new MvQMenuItem(actionAbout,MvQMenuItem::MenuTarget));
}

void FlextraExaminer::setupLogArea()
{
	// Log Area
	QVBoxLayout *logLayout = new QVBoxLayout;
	logLayout->setObjectName(QString::fromUtf8("vboxLayout"));
	logLayout->setContentsMargins(0,0,0,0);
	logLayout->setSpacing(1);
	logPanel_= new QWidget;
	logPanel_->setMinimumHeight(100);
	logPanel_->setLayout(logLayout);
	//mainSplitter_->addWidget(logPanel_);

	// Define header layout
	QHBoxLayout *hb = new QHBoxLayout;
	hb->setContentsMargins(0,0,0,0);
	logLayout->addLayout(hb);

	// Add label to the header
	QLabel* label = new QLabel(tr("Log"));
	label->setFrameShape(QFrame::StyledPanel);
	hb->addWidget(label,1);

	// Add Clear button to the header
	QPushButton *clearPb = new QPushButton(tr("Clear log"));
	hb->addWidget(clearPb);

	// Log browser
	logBrowser_= new MvQLogBrowser(logLayout);

	// Signals and slots
	connect ( clearPb,SIGNAL(clicked(bool)), this,SLOT(slotClearLog(bool)) );
}

void FlextraExaminer::setupDumpPanel()
{	
        // Build a layout
        QVBoxLayout* dumpLayout = new QVBoxLayout;
        dumpLayout->setContentsMargins(0,0,0,0);
        dumpPanel_ = new QWidget;
        dumpPanel_->setLayout(dumpLayout);
            
        dumpText_ = new QPlainTextEdit;
        dumpText_->setReadOnly(true);
        dumpText_->setLineWrapMode(QPlainTextEdit::NoWrap);
        //dumpText_->setProperty("mvStyle","fileContents");
 
	//We need a fixedpitch font!!
	QFont font=dumpText_->font();
        font.setFamily("Courier");
        font.setFixedPitch(true);
	font.setPointSize(12);
	dumpText_->setFont(font);
		
	MvQHighlighter* hl;
	hl=new MvQHighlighter(dumpText_->document());
	
	dumpLayout->addWidget(dumpText_);

        //Build the search widget
        findPanel_= new MvQPlainTextEditSearchLine(dumpText_,tr("&Find:"));
        dumpLayout->addWidget(findPanel_);
	findPanel_->hide();	
}
	
void FlextraExaminer::setupBlockPanel()
{			
	QVBoxLayout* blockLayout = new QVBoxLayout;
	blockLayout->setContentsMargins(0,0,0,0);
	blockPanel_ = new QWidget;
	blockPanel_->setLayout(blockLayout);
	
	// Build a tree to show the data representation
        blockTree_ = new QTreeWidget;
        blockTree_->setObjectName("ncTreeParam");
        blockTree_->setProperty("mvStyle",0);
        blockTree_->setAlternatingRowColors(true);
        blockTree_->setAllColumnsShowFocus(true);
	blockTree_->setColumnCount(4);
        blockTree_->setUniformRowHeights(true);
	blockTree_->setRootIsDecorated(false);
	
	QStringList hdr;
	hdr << tr("Index") << tr("Name") << tr("Step type") << tr("U");
	blockTree_->setHeaderLabels(hdr);
	
	
	QTreeWidgetItem *header=blockTree_->headerItem();
	header->setData(3,Qt::ToolTipRole,tr("Uncertainty trajectory"));
	
	blockTree_->setContextMenuPolicy(Qt::CustomContextMenu);
	
	blockLayout->addWidget(blockTree_);
	
	//connect(blockTree_,SIGNAL(itemActivated(QTreeWidgetItem*,int)),
	//	this,SLOT(slotBlockSelected(QTreeWidgetItem*,int)));
		
	connect(blockTree_,SIGNAL(customContextMenuRequested(const QPoint &)),
                this,SLOT(slotContextMenu(const QPoint &)));	
}


bool FlextraExaminer::initData(const string& filename,const string& icon)
{
	iconPath_=icon;
  
  	// Clean data if already exists
	if ( data_ )
	{
		delete data_;
		data_ = 0;
	}

	// Write initial message in the log area
	LogItem *log = new LogItem;
	LogHandler::instance()->add(log);
	log->description("Loading file");
	log->method("MvFlextra API");

	// Initialize Flextra object
	data_ = new MvFlextra(filename);
	
	// Update File info label
	updateFileInfoLabel();
	
	if(data_->blockNum() == 0)
	{
	  	delete data_;
		data_ = 0;
		log->error("Could not open geopoints file");
		slotUpdateLogOutput();
		QString buf = tr("<b>Status: </b><font color=red>FAILED</font>");
		emit statusMessage(buf);
		return false;
	}   
	else if(data_->blockNum() == 1)
	{
		QFile file(QString::fromStdString(filename));
		if(file.open(QIODevice::ReadOnly | QIODevice::Text))
		{
			QString str,line;
			QTextStream in(&file);
    			while(!in.atEnd())
    			{
        			line=in.readLine();
				str+=line+'\n';				
    			}		  		  		  
			dumpText_->setPlainText(str); 
		}
		file.close();
		blockPanel_->hide();
	}	
	else
	{
		QTreeWidgetItem *firstItem=0;
		
		for(int i=0; i < data_->blockNum(); i++)
		{
			QTreeWidgetItem *item = new QTreeWidgetItem(blockTree_);			
			QString comment=QString::fromStdString(data_->blocks().at(i)->comment());			
			QString step=(data_->blocks().at(i)->constantStep())?tr("Constant"):tr("Flexible");
			QString utr=(data_->blocks().at(i)->uncertaintyTr())?tr("y"):tr("n"); 
			
			item->setData(0,Qt::DisplayRole,QString::number(i+1));
			item->setData(1,Qt::DisplayRole,comment);
			item->setData(2,Qt::DisplayRole,step);
			item->setData(3,Qt::DisplayRole,utr);
			
			if(!firstItem)
			  	firstItem=item;
		}
		blockTree_->resizeColumnToContents(0);
		blockTree_->resizeColumnToContents(1); 
		blockTree_->resizeColumnToContents(2); 			
		
		connect(blockTree_,SIGNAL(currentItemChanged(QTreeWidgetItem*,QTreeWidgetItem*)),
			this,SLOT(slotBlockSelected(QTreeWidgetItem*,QTreeWidgetItem*)));
		
		if(firstItem)
		{  	
		  	blockTree_->setCurrentItem(firstItem);
		}	
	}
	
	
	slotUpdateLogOutput();
	QString buf = tr("<b>Status:</b> <font color=green>OK</font>");
	emit statusMessage(buf);

	return true;
}

void FlextraExaminer::slotUpdateLogOutput()
{
	logBrowser_->update();
}

void FlextraExaminer::slotClearLog(bool)
{
	LogHandler::instance()->clear();
	logBrowser_->clear();
}

void FlextraExaminer::slotStatusMessage(QString s)
{
	statusMessageLabel_->setText(s);
}

void FlextraExaminer::updateFileInfoLabel()
{
	fileInfoLabel_->setFlextraTextLabel(QString::fromStdString(data_->fileName()),QString::fromStdString(iconPath_),data_->blockNum());
}

void FlextraExaminer::slotShowAboutBox()
{
	MvQAbout about("FLEXTRA Examiner","",MvQAbout::MetviewVersion );
	about.exec();
}

void FlextraExaminer::slotBlockSelected(QTreeWidgetItem* item,QTreeWidgetItem*)
{
	int row=item->data(0,Qt::DisplayRole).toString().toInt();	  
	row--;
	if(row >= 0 && row < data_->blockNum())
	{
		if(tmpFilePath_.empty())
		{
			tmpFilePath_=string(marstmp());
			
		}
		data_->write(tmpFilePath_,row);
		
		QFile file(QString::fromStdString(tmpFilePath_));
		if(file.open(QIODevice::ReadOnly | QIODevice::Text))
		{
			QString str,line;
			QTextStream in(&file);
    			while(!in.atEnd())
    			{
        			line=in.readLine();
				str+=line+'\n';				
    			}		  		  		  
			dumpText_->setPlainText(str); 
		}
		file.close();
	}
	else
	{
		dumpText_->clear();
	}	
}  


void FlextraExaminer::slotContextMenu(const QPoint &position)
{
	QTreeWidgetItem *item=blockTree_->itemAt(position); 
	int row=item->data(0,Qt::DisplayRole).toString().toInt();	  
	row--;
	if(row < 0 || row >= data_->blockNum())
	 	return; 
	  
	QList<QAction*> actions;

	//Inser new key
	QAction *actionSave = new QAction(this);
        actionSave->setObjectName(QString::fromUtf8("actionSave"));
	actionSave->setText(tr("Save"));
	actions.append(actionSave);

	if(QMenu::exec(actions,blockTree_->mapToGlobal(position)) == actionSave) 
	{
		QString offeredName=QString::fromStdString(iconPath_).section('/',0,-2)  + 
		                    "/res_" + item->data(1,Qt::DisplayRole).toString() + ".txt";		
	  	QString fileName = QFileDialog::getSaveFileName(this,
     				tr("Save Data To File"),offeredName, QString());  
	  					
		data_->write(fileName.toStdString(),row);						
	}  
		
	delete actionSave;
}
	
void FlextraExaminer::writeSettings()
{
	QSettings settings("ECMWF","MV4-FlextraExaminer");
	
	settings.beginGroup("mainWindow");
	settings.setValue("geometry",saveGeometry());
	settings.setValue("centralSplitter",centralSplitter_->saveState());
	settings.setValue("mainSplitter",mainSplitter_->saveState());
	settings.setValue("actionFileInfoStatus",actionFileInfo_->isChecked());
	settings.endGroup();
}

void FlextraExaminer::readSettings()
{
	QSettings settings("ECMWF","MV4-FlextraExaminer");

	QVariant value;
	settings.beginGroup("mainWindow");
	restoreGeometry(settings.value("geometry").toByteArray());
	centralSplitter_->restoreState(settings.value("centralSplitter").toByteArray());
	mainSplitter_->restoreState(settings.value("mainSplitter").toByteArray());
	
	if(settings.value("actionFileInfoStatus").isNull())
		actionFileInfo_->setChecked(true);
	else
		actionFileInfo_->setChecked(settings.value("actionFileInfoStatus").toBool());

	settings.endGroup();
}
