#
# convertor.py : LiveImageConvertor and LiveUSBImageConvertor Implementation
#
# Copyright 2009, Intel Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import os
import os.path
import sys
import time
import logging
import shutil
import subprocess
import string
import glob
import rpm

import mic.appcreate as appcreate
import mic.imgcreate as imgcreate
import mic.imgcreate.pkgmanagers as pkgmanagers

class ImageConvertor:
    def __init__(self):
        self._srcloop = None
        self._srcimage = None
        self._srcimgsize = 0
        self._srcfstype = "ext3"
        self._srcblocksize = 4096
        self._tmpimgpath = None
        self._srcmnt = None
        self._srcfmt = None

    def set_source_image(self, path, fmt):
        if not os.path.exists(path):
            raise CreatorError("Source image %s doesn't exist." % path)
        self._srcfmt = fmt
        if fmt in ("vmdk", "vdi"):
            dstfmt = "raw"
            dstimg = "%s/%s.%s" % (self._tmpimgpath, os.path.basename(path).replace(".", "-"), dstfmt)
            imgcreate.convert_image(path, fmt, dstimg, dstfmt)
            path = dstimg
            self._srcfmt = dstfmt

        self._srcimage = path
        if os.path.isdir(path):
            print "Counting file system size..."
        self._srcimgsize = (imgcreate.get_file_size(path)) * 1024L * 1024L
        if os.path.isdir(path):
            self._srcimgsize += 256 * 1024L * 1024L
        self._set_fstype("ext3")
        self._set_image_size(self._srcimgsize)

    def _mount_srcimg(self):
        if self._srcfmt in ("livecd", "liveusb", "fs", "ext3fsimg"):
            return
        disk = imgcreate.SparseLoopbackDisk(self._srcimage, self._srcimgsize)
        self._srcloop = appcreate.PartitionedMount({'/dev/sdb':disk}, self._srcmnt, skipformat = True)

        self._srcloop.add_partition(self._srcimgsize/1024/1024, "/dev/sdb", "/", self._srcfstype, boot=False)
        try:
            self._srcloop.mount()
        except imgcreate.MountError, e:
            self._srcloop.cleanup()
            raise imgcreate.CreatorError("Failed to loopback mount '%s' : %s" %
                               (self._srcimage, e))

    def _unmount_srcimg(self):
        if self._srcfmt in ("livecd", "liveusb", "fs", "ext3fsimg"):
            return
        if self._srcloop:
            self._srcloop.cleanup()

    def _select_kernel_pkg(self, pkg_manager):
        kernel_pkgs = []
        for pkg in imgcreate.kickstart.get_packages(self.ks):
            if pkg.startswith('kernel'):
                kernel_pkgs.append(pkg)
        if not kernel_pkgs:
            kernel_pkgs.append('kernel')

        for pkg in kernel_pkgs:
            e = pkg_manager.selectPackage(pkg)
            if e:
                if imgcreate.kickstart.ignore_missing(self.ks):
                    skipped_pkgs.append(pkg)
                else:
                    raise imgcreate.CreatorError("Failed to find package '%s' : %s" %
                                       (pkg, e))

    def get_kernel_versions(self, instroot):
        ret = {}
        files = glob.glob(instroot + "/boot/vmlinuz-*")
        for file in files:
            version = os.path.basename(file)[8:]
            if version is None:
                continue

            name = "kernel"
            if not name in ret:
                ret[name] = [version]
            elif not version in ret[name]:
                ret[name].append(version)

        return ret

    def get_kernel_options(self, instroot):
        r = "ro liveimg quiet"
        if os.path.exists(instroot + "/usr/bin/rhgb"):
            r += " rhgb"
        if os.path.exists(instroot + "/usr/bin/plymouth"):
            r += " rhgb"
        return r

    def get_menu_args(self):
        return "bootinstall autoinst"

    def _reinstall_kernel(self, kernel, instroot, repo_urls = {}):
        # Remove current kernel
        args = [ imgcreate.fs.find_binary_path("rpm"), "--root=" + instroot, "-e", kernel ]
        subprocess.call(args)

        pkg_manager = self.get_pkg_manager()
        pkg_manager.setup(self._yum_conf, self._instroot)

        for repo in imgcreate.kickstart.get_repos(self.ks, repo_urls):
            (name, baseurl, mirrorlist, inc, exc, proxy, proxy_username, proxy_password, debuginfo, source, gpgkey) = repo

            yr = pkg_manager.addRepository(name, baseurl, mirrorlist, proxy, proxy_username, proxy_password)
            if inc:
                yr.includepkgs = inc
            if exc:
                yr.exclude = exc

        rpm.addMacro("_excludedocs", "1")
        rpm.addMacro("__file_context_path", "%{nil}")

        try:
            try:
                self._select_kernel_pkg(pkg_manager)
                pkg_manager.runInstall()
            except imgcreate.CreatorError, e:
                raise imgcreate.CreatorError("%s" % (e,))
        finally:
            pkg_manager.closeRpmDB()
            pkg_manager.close()
            os.unlink(self._yum_conf)

    def _copyfs(self, src, dst):
        if self._srcfmt == "livecd" or self._srcfmt == "liveusb":
            return
        print "Copying filesystem..."
        imgcreate.myxcopytree(src, dst)


class LiveImageConvertor(imgcreate.LiveImageCreator, ImageConvertor):
    def __init__(self, *args):
        ImageConvertor.__init__(self)
        imgcreate.LiveImageCreator.__init__(self, *args)

    def setup_temp_dirs(self, tmpdir):
        self.tmpdir = tmpdir
        self._tmpimgpath = self._mkdtemp("raw-image-")
        self._srcmnt = self._mkdtemp("srcmnt")
        self._yum_conf = self._mktemp(prefix = "yum.conf-")

    def _mount_instroot(self, base_on = None):
        self._mount_srcimg()
        if self._srcfmt == "raw" and not base_on:
            base_on = self._srcloop.partitions[0]['device']
            self.actasconvertor = True
        if self._srcfmt == "ext3fsimg":
            self.actasconvertor = True
        imgcreate.LiveImageCreator._mount_instroot(self, base_on)

    def _unmount_instroot(self):
        self._unmount_srcimg()
        imgcreate.LiveImageCreator._unmount_instroot(self)

    def install(self, repo_urls = {}):
        if os.path.isdir(self._srcimage):
            self._copyfs(self._srcimage, self._instroot)
        #kernel = self._get_kernel_versions().keys()[0]
        #self._reinstall_kernel(kernel, self._instroot)
        # Install additional packages here
        self._mkinitrd()

    def __base_on(self, base_on):
        """Support Image Convertor"""
        if self.actasconvertor:
            if os.path.exists(base_on) and not os.path.isfile(base_on):
                args = [ "/bin/dd", "if=%s" % base_on, "of=%s" % self._image ]
                print "dd %s -> %s" % (base_on, self._image)
                rc = subprocess.call(args)
                if rc != 0:
                    raise CreatorError("Failed to dd from %s to %s" % (base_on, self._image))
                self._set_image_size(imgcreate.get_file_size(self._image) * 1024L * 1024L)
            return

        """helper function to extract ext3 file system from a live usb image"""
        usbimgsize = imgcreate.get_file_size(base_on) * 1024L * 1024L
        disk = imgcreate.SparseLoopbackDisk(base_on, usbimgsize)
        usbimgmnt = self._mkdtemp("usbimgmnt-")
        usbloop = appcreate.PartitionedMount({'/dev/sdb':disk}, usbimgmnt, skipformat = True)
        usbimg_fstype = "vfat"
        usbloop.add_partition(usbimgsize/1024/1024, "/dev/sdb", "/", usbimg_fstype, boot=False)
        try:
            usbloop.mount()
        except imgcreate.MountError, e:
            usbloop.cleanup()
            raise imgcreate.CreatorError("Failed to loopback mount '%s' : %s" %
                               (base_on, e))

        # legacy LiveOS filesystem layout support, remove for F9 or F10
        if os.path.exists(usbimgmnt + "/squashfs.img"):
            squashimg = usbimgmnt + "/squashfs.img"
        else:
            squashimg = usbimgmnt + "/LiveOS/squashfs.img"

        tmpoutdir = self._mkdtemp()
        # unsquashfs requires outdir mustn't exist
        shutil.rmtree(tmpoutdir, ignore_errors = True)
        self._uncompress_squashfs(squashimg, tmpoutdir)

        try:
            # legacy LiveOS filesystem layout support, remove for F9 or F10
            if os.path.exists(tmpoutdir + "/os.img"):
                os_image = tmpoutdir + "/os.img"
            else:
                os_image = tmpoutdir + "/LiveOS/ext3fs.img"

            if not os.path.exists(os_image):
                raise imgcreate.CreatorError("'%s' is not a valid live CD ISO : neither "
                                   "LiveOS/ext3fs.img nor os.img exist" %
                                   base_on)

            print "Copying file system..."
            shutil.copyfile(os_image, self._image)
            self._set_image_size(imgcreate.get_file_size(self._image) * 1024L * 1024L)
        finally:
            shutil.rmtree(tmpoutdir, ignore_errors = True)
            usbloop.cleanup()

    def _base_on(self, base_on):
        if self._srcfmt == "liveusb":
            self.__base_on(base_on)
        else:
            imgcreate.LiveImageCreator._base_on(self, base_on)

    def _mkinitrd(self):
        kernelver = self.get_kernel_versions(self._instroot).values()[0][0]
        args = [ "/usr/libexec/mkliveinitrd", "/boot/initrd-%s.img" % kernelver, "%s" % kernelver ]
        try:
            subprocess.call(args, preexec_fn = self._chroot)
        except OSError, (err, msg):
           raise imgcreate.CreatorError("Failed to execute /usr/libexec/mkliveinitrd: %s" % msg)

    def _run_post_cleanups(self):
        kernelver = self.get_kernel_versions(self._instroot).values()[0][0]
        args = ["rm", "-f", "/boot/initrd-%s.img" % kernelver]
        try:
            subprocess.call(args, preexec_fn = self._chroot)
        except OSError, (err, msg):
           raise imgcreate.CreatorError("Failed to run post cleanups: %s" % msg)

    def configure(self):
        self._create_bootconfig()
        self._run_post_cleanups()

    def _get_kernel_versions(self):
        return self.get_kernel_versions(self._instroot)

    def _get_kernel_options(self):
        return self.get_kernel_options(self._instroot)

    def _get_menu_options(self):
        """Return a menu options string for syslinux configuration."""
        return self.get_menu_args()

class LiveUSBImageConvertor(imgcreate.LiveUSBImageCreator, ImageConvertor):
    def __init__(self, *args):
        ImageConvertor.__init__(self)
        imgcreate.LiveUSBImageCreator.__init__(self, *args)

    def setup_temp_dirs(self, tmpdir):
        self.tmpdir = tmpdir
        self._tmpimgpath = self._mkdtemp("raw-image-")
        self._srcmnt = self._mkdtemp("srcmnt")
        self._yum_conf = self._mktemp(prefix = "yum.conf-")

    def _mount_instroot(self, base_on = None):
        self._mount_srcimg()
        if self._srcfmt == "raw" and not base_on:
            base_on = self._srcloop.partitions[0]['device']
            self.actasconvertor = True
        if self._srcfmt == "ext3fsimg":
            self.actasconvertor = True
        imgcreate.LiveUSBImageCreator._mount_instroot(self, base_on)

    def _unmount_instroot(self):
        self._unmount_srcimg()
        imgcreate.LiveUSBImageCreator._unmount_instroot(self)

    def install(self, repo_urls = {}):
        if os.path.isdir(self._srcimage):
            self._copyfs(self._srcimage, self._instroot)
        #kernel = self._get_kernel_versions().keys()[0]
        #self._reinstall_kernel(kernel, self._instroot)
        self._mkinitrd()

    def _base_on(self, base_on):
        if self._srcfmt == "liveusb":
            imgcreate.LiveUSBImageCreator._base_on(self, base_on)
        else:
            imgcreate.LiveImageCreator._base_on(self, base_on)

    def _mkinitrd(self):
        kernelver = self.get_kernel_versions(self._instroot).values()[0][0]
        args = [ "/usr/libexec/mkliveinitrd", "/boot/initrd-%s.img" % kernelver, "%s" % kernelver ]
        try:
            subprocess.call(args, preexec_fn = self._chroot)
        except OSError, (err, msg):
           raise imgcreate.CreatorError("Failed to execute /usr/libexec/mkliveinitrd: %s" % msg)

    def _run_post_cleanups(self):
        kernelver = self.get_kernel_versions(self._instroot).values()[0][0]
        args = ["rm", "-f", "/boot/initrd-%s.img" % kernelver]
        try:
            subprocess.call(args, preexec_fn = self._chroot)
        except OSError, (err, msg):
           raise imgcreate.CreatorError("Failed to run post cleanups: %s" % msg)

    def configure(self):
        self._create_bootconfig()
        self._run_post_cleanups()

    def _get_kernel_versions(self):
        return self.get_kernel_versions(self._instroot)

    def _get_kernel_options(self):
        return self.get_kernel_options(self._instroot)

    def _get_menu_options(self):
        """Return a menu options string for syslinux configuration."""
        return self.get_menu_args()
