/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "common/config-manager.h"
#include "common/translation.h"
#include "gui/gui-manager.h"
#include "gui/message.h"
#include "graphics/fonts/amigafont.h"

#include "agi/agi.h"
#include "agi/font.h"
#include "agi/text.h"

namespace Agi {

GfxFont::GfxFont(AgiBase *vm) {
	_vm = vm;

	_fontData = nullptr;
	_fontDataAllocated = nullptr;
	_fontIsHires = false;
}

GfxFont::~GfxFont() {
	free(_fontDataAllocated);
}

// Arrow to the right character, used for original saved game dialogs
// Needs to get patched into at least the Apple IIgs font, because the font didn't support
// that character and original AGI on Apple IIgs used Apple II menus for saving/restoring.
static const uint8 fontData_ArrowRightCharacter[8] = {
	0x00, 0x18, 0x0C, 0xFE, 0x0C, 0x18, 0x00, 0x00,
};

// 8x8 font patterns

// this is basically the standard PC BIOS font, taken from Dos-Box, with a few modifications
static const uint8 fontData_PCBIOS[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7E, 0x81, 0xA5, 0x81, 0xBD, 0x99, 0x81, 0x7E,
	0x7E, 0xFF, 0xDB, 0xFF, 0xC3, 0xE7, 0xFF, 0x7E,
	0x6C, 0xFE, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00,
	0x10, 0x38, 0x7C, 0xFE, 0x7C, 0x38, 0x10, 0x00,
	0x38, 0x7C, 0x38, 0xFE, 0xFE, 0x7C, 0x38, 0x7C,
	0x10, 0x10, 0x38, 0x7C, 0xFE, 0x7C, 0x38, 0x7C,
	0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00,
	0xFF, 0xFF, 0xE7, 0xC3, 0xC3, 0xE7, 0xFF, 0xFF,
	0x00, 0x3C, 0x66, 0x42, 0x42, 0x66, 0x3C, 0x00,
	0xFF, 0xC3, 0x99, 0xBD, 0xBD, 0x99, 0xC3, 0xFF,
	0x0F, 0x07, 0x0F, 0x7D, 0xCC, 0xCC, 0xCC, 0x78,
	0x3C, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x7E, 0x18,
	0x08, 0x0C, 0x0A, 0x0A, 0x08, 0x78, 0xF0, 0x00, // 0x0D changed
	0x18, 0x14, 0x1A, 0x16, 0x72, 0xE2, 0x0E, 0x1C, // 0x0E changed
	0x10, 0x54, 0x38, 0xEE, 0x38, 0x54, 0x10, 0x00, // 0x0F changed
	//0x3F, 0x33, 0x3F, 0x30, 0x30, 0x70, 0xF0, 0xE0, // 0x0D original
	//0x7F, 0x63, 0x7F, 0x63, 0x63, 0x67, 0xE6, 0xC0, // 0x0E original
	//0x99, 0x5A, 0x3C, 0xE7, 0xE7, 0x3C, 0x5A, 0x99, // 0x0F original
	0x80, 0xE0, 0xF8, 0xFE, 0xF8, 0xE0, 0x80, 0x00,
	0x02, 0x0E, 0x3E, 0xFE, 0x3E, 0x0E, 0x02, 0x00,
	0x18, 0x3C, 0x5A, 0x18, 0x5A, 0x3C, 0x18, 0x00, // 0x12 changed
	//0x18, 0x3C, 0x7E, 0x18, 0x18, 0x7E, 0x3C, 0x18, // 0x12 original
	0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00,
	0x7F, 0xDB, 0xDB, 0xDB, 0x7B, 0x1B, 0x1B, 0x00, // 0x14 changed
	0x1C, 0x22, 0x38, 0x44, 0x44, 0x38, 0x88, 0x70, // 0x14 changed
	//0x7F, 0xDB, 0xDB, 0x7B, 0x1B, 0x1B, 0x1B, 0x00, // 0x14 original
	//0x3E, 0x63, 0x38, 0x6C, 0x6C, 0x38, 0xCC, 0x78, // 0x15 original
	0x00, 0x00, 0x00, 0x00, 0x7E, 0x7E, 0x7E, 0x00,
	0x18, 0x3C, 0x5A, 0x18, 0x5A, 0x3C, 0x18, 0x7E, // 0x17 changed
	0x18, 0x3C, 0x5A, 0x18, 0x18, 0x18, 0x18, 0x00, // 0x18 changed
	0x18, 0x18, 0x18, 0x18, 0x5A, 0x3C, 0x18, 0x00, // 0x19 changed
	//0x18, 0x3C, 0x7E, 0x18, 0x7E, 0x3C, 0x18, 0xFF, // 0x17 original
	//0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x00, // 0x18 original
	//0x18, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, // 0x19 original
	0x00, 0x18, 0x0C, 0xFE, 0x0C, 0x18, 0x00, 0x00,
	0x00, 0x30, 0x60, 0xFE, 0x60, 0x30, 0x00, 0x00,
	0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xFE, 0x00, 0x00,
	0x00, 0x24, 0x42, 0xFF, 0x42, 0x24, 0x00, 0x00, // 0x1D changed
	0x00, 0x10, 0x38, 0x7C, 0xFE, 0xFE, 0x00, 0x00, // 0x1E changed
	0x00, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x00, // 0x1F changed
	//0x00, 0x24, 0x66, 0xFF, 0x66, 0x24, 0x00, 0x00, // 0x1D original
	//0x00, 0x18, 0x3C, 0x7E, 0xFF, 0xFF, 0x00, 0x00, // 0x1E original
	//0x00, 0xFF, 0xFF, 0x7E, 0x3C, 0x18, 0x00, 0x00, // 0x1F original
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20
	0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00,
	0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x6C, 0x6C, 0xFE, 0x6C, 0xFE, 0x6C, 0x6C, 0x00,
	0x30, 0x7C, 0xC0, 0x78, 0x0C, 0xF8, 0x30, 0x00,
	0x00, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xC6, 0x00,
	0x38, 0x6C, 0x38, 0x76, 0xDC, 0xCC, 0x76, 0x00,
	0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00,
	0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00,
	0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00,
	0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x80, 0x00,
	0x7C, 0xC6, 0xCE, 0xDE, 0xF6, 0xE6, 0x7C, 0x00, // 0x30
	0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xFC, 0x00,
	0x78, 0xCC, 0x0C, 0x38, 0x60, 0xCC, 0xFC, 0x00,
	0x78, 0xCC, 0x0C, 0x38, 0x0C, 0xCC, 0x78, 0x00,
	0x1C, 0x3C, 0x6C, 0xCC, 0xFE, 0x0C, 0x1E, 0x00,
	0xFC, 0xC0, 0xF8, 0x0C, 0x0C, 0xCC, 0x78, 0x00,
	0x38, 0x60, 0xC0, 0xF8, 0xCC, 0xCC, 0x78, 0x00,
	0xFC, 0xCC, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0xCC, 0x78, 0xCC, 0xCC, 0x78, 0x00,
	0x78, 0xCC, 0xCC, 0x7C, 0x0C, 0x18, 0x70, 0x00,
	0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x00, 0x30, 0x30, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x18, 0x30, 0x60, 0xC0, 0x60, 0x30, 0x18, 0x00,
	0x00, 0x00, 0xFC, 0x00, 0x00, 0xFC, 0x00, 0x00,
	0x60, 0x30, 0x18, 0x0C, 0x18, 0x30, 0x60, 0x00,
	0x78, 0xCC, 0x0C, 0x18, 0x30, 0x00, 0x30, 0x00,
	0x7C, 0xC6, 0xDE, 0xDE, 0xDE, 0xC0, 0x78, 0x00, // 0x40
	0x30, 0x78, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0x00,
	0xFC, 0x66, 0x66, 0x7C, 0x66, 0x66, 0xFC, 0x00,
	0x3C, 0x66, 0xC0, 0xC0, 0xC0, 0x66, 0x3C, 0x00,
	0xF8, 0x6C, 0x66, 0x66, 0x66, 0x6C, 0xF8, 0x00,
	0xFE, 0x62, 0x68, 0x78, 0x68, 0x62, 0xFE, 0x00,
	0xFE, 0x62, 0x68, 0x78, 0x68, 0x60, 0xF0, 0x00,
	0x3C, 0x66, 0xC0, 0xC0, 0xCE, 0x66, 0x3E, 0x00,
	0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00,
	0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x1E, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78, 0x00,
	0xE6, 0x66, 0x6C, 0x78, 0x6C, 0x66, 0xE6, 0x00,
	0xF0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xFE, 0x00,
	0xC6, 0xEE, 0xFE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00,
	0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00,
	0x38, 0x6C, 0xC6, 0xC6, 0xC6, 0x6C, 0x38, 0x00,
	0xFC, 0x66, 0x66, 0x7C, 0x60, 0x60, 0xF0, 0x00, // 0x50
	0x78, 0xCC, 0xCC, 0xCC, 0xDC, 0x78, 0x1C, 0x00,
	0xFC, 0x66, 0x66, 0x7C, 0x6C, 0x66, 0xE6, 0x00,
	0x78, 0xCC, 0xE0, 0x70, 0x1C, 0xCC, 0x78, 0x00,
	0xFC, 0xB4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xFC, 0x00,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00,
	0xC6, 0xC6, 0xC6, 0xD6, 0xFE, 0xEE, 0xC6, 0x00,
	0xC6, 0xC6, 0x6C, 0x38, 0x38, 0x6C, 0xC6, 0x00,
	0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x78, 0x00,
	0xFE, 0xC6, 0x8C, 0x18, 0x32, 0x66, 0xFE, 0x00,
	0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x02, 0x00,
	0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00,
	0x10, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x60
	0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x76, 0x00,
	0xE0, 0x60, 0x60, 0x7C, 0x66, 0x66, 0xDC, 0x00,
	0x00, 0x00, 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x00,
	0x1C, 0x0C, 0x0C, 0x7C, 0xCC, 0xCC, 0x76, 0x00,
	0x00, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0x38, 0x6C, 0x60, 0xF0, 0x60, 0x60, 0xF0, 0x00,
	0x00, 0x00, 0x76, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8,
	0xE0, 0x60, 0x6C, 0x76, 0x66, 0x66, 0xE6, 0x00,
	0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x0C, 0x00, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78,
	0xE0, 0x60, 0x66, 0x6C, 0x78, 0x6C, 0xE6, 0x00,
	0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x00, 0x00, 0xCC, 0xFE, 0xFE, 0xD6, 0xC6, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00,
	0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0xDC, 0x66, 0x66, 0x7C, 0x60, 0xF0, // 0x70
	0x00, 0x00, 0x76, 0xCC, 0xCC, 0x7C, 0x0C, 0x1E,
	0x00, 0x00, 0xDC, 0x76, 0x66, 0x60, 0xF0, 0x00,
	0x00, 0x00, 0x7C, 0xC0, 0x78, 0x0C, 0xF8, 0x00,
	0x10, 0x30, 0x7C, 0x30, 0x30, 0x34, 0x18, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00,
	0x00, 0x00, 0xC6, 0xD6, 0xFE, 0xFE, 0x6C, 0x00,
	0x00, 0x00, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8,
	0x00, 0x00, 0xFC, 0x98, 0x30, 0x64, 0xFC, 0x00,
	0x1C, 0x30, 0x30, 0xE0, 0x30, 0x30, 0x1C, 0x00,
	0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00,
	0xE0, 0x30, 0x30, 0x1C, 0x30, 0x30, 0xE0, 0x00,
	0x76, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00,
	0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x18, 0x0C, 0x78, // 0x80
	0x00, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00,
	0x1C, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0x7E, 0xC3, 0x3C, 0x06, 0x3E, 0x66, 0x3F, 0x00,
	0xCC, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00,
	0xE0, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00,
	0x30, 0x30, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00,
	0x00, 0x00, 0x78, 0xC0, 0xC0, 0x78, 0x0C, 0x38,
	0x7E, 0xC3, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0xCC, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0xE0, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0xCC, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x7C, 0xC6, 0x38, 0x18, 0x18, 0x18, 0x3C, 0x00,
	0xE0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
	0xC6, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x30, 0x30, 0x00, 0x78, 0xCC, 0xFC, 0xCC, 0x00,
	0x1C, 0x00, 0xFC, 0x60, 0x78, 0x60, 0xFC, 0x00,
	0x00, 0x00, 0x7F, 0x0C, 0x7F, 0xCC, 0x7F, 0x00,
	0x3E, 0x6C, 0xCC, 0xFE, 0xCC, 0xCC, 0xCE, 0x00,
	0x78, 0xCC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0xCC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0xE0, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00,
	0x78, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00,
	0x00, 0xE0, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00,
	0x00, 0xCC, 0x00, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8,
	0xC3, 0x18, 0x3C, 0x66, 0x66, 0x3C, 0x18, 0x00,
	0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x18, 0x18, 0x7E, 0xC0, 0xC0, 0x7E, 0x18, 0x18,
	0x38, 0x6C, 0x64, 0xF0, 0x60, 0xE6, 0xFC, 0x00,
	0xCC, 0xCC, 0x78, 0xFC, 0x30, 0xFC, 0x30, 0x30,
	0xF8, 0xCC, 0xCC, 0xFA, 0xC6, 0xCF, 0xC6, 0xC7,
	0x0E, 0x1B, 0x18, 0x3C, 0x18, 0x18, 0xD8, 0x70,
	0x1C, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00,
	0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x00, 0x1C, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x1C, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00,
	0x00, 0xF8, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0x00,
	0xFC, 0x00, 0xCC, 0xEC, 0xFC, 0xDC, 0xCC, 0x00,
	0x3C, 0x6C, 0x6C, 0x3E, 0x00, 0x7E, 0x00, 0x00,
	0x38, 0x6C, 0x6C, 0x38, 0x00, 0x7C, 0x00, 0x00,
	0x30, 0x00, 0x30, 0x60, 0xC0, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0x00, 0xFC, 0xC0, 0xC0, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xFC, 0x0C, 0x0C, 0x00, 0x00,
	0xC3, 0xC6, 0xCC, 0xDE, 0x33, 0x66, 0xCC, 0x0F,
	0xC3, 0xC6, 0xCC, 0xDB, 0x37, 0x6F, 0xCF, 0x03,
	0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x33, 0x66, 0xCC, 0x66, 0x33, 0x00, 0x00,
	0x00, 0xCC, 0x66, 0x33, 0x66, 0xCC, 0x00, 0x00,
	0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88,
	0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA,
	0xDB, 0x77, 0xDB, 0xEE, 0xDB, 0x77, 0xDB, 0xEE,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0xF8, 0x18, 0x18, 0x18,
	0x18, 0x18, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18,
	0x36, 0x36, 0x36, 0x36, 0xF6, 0x36, 0x36, 0x36,
	0x00, 0x00, 0x00, 0x00, 0xFE, 0x36, 0x36, 0x36,
	0x00, 0x00, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18,
	0x36, 0x36, 0xF6, 0x06, 0xF6, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x00, 0x00, 0xFE, 0x06, 0xF6, 0x36, 0x36, 0x36,
	0x36, 0x36, 0xF6, 0x06, 0xFE, 0x00, 0x00, 0x00,
	0x36, 0x36, 0x36, 0x36, 0xFE, 0x00, 0x00, 0x00,
	0x18, 0x18, 0xF8, 0x18, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xF8, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x1F, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x18, 0x18, 0xFF, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x1F, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x18, 0x18, 0xFF, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18,
	0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x37, 0x30, 0x3F, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x3F, 0x30, 0x37, 0x36, 0x36, 0x36,
	0x36, 0x36, 0xF7, 0x00, 0xFF, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xFF, 0x00, 0xF7, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36,
	0x00, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00,
	0x36, 0x36, 0xF7, 0x00, 0xF7, 0x36, 0x36, 0x36,
	0x18, 0x18, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00,
	0x36, 0x36, 0x36, 0x36, 0xFF, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xFF, 0x00, 0xFF, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x3F, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x1F, 0x18, 0x1F, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x00, 0x3F, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0xFF, 0x36, 0x36, 0x36,
	0x18, 0x18, 0xFF, 0x18, 0xFF, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0xF8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x1F, 0x18, 0x18, 0x18,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x76, 0xDC, 0xC8, 0xDC, 0x76, 0x00,
	0x00, 0x78, 0xCC, 0xF8, 0xCC, 0xF8, 0xC0, 0xC0,
	0x00, 0xFC, 0xCC, 0xC0, 0xC0, 0xC0, 0xC0, 0x00,
	0x00, 0xFE, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x00,
	0xFC, 0xCC, 0x60, 0x30, 0x60, 0xCC, 0xFC, 0x00,
	0x00, 0x00, 0x7E, 0xD8, 0xD8, 0xD8, 0x70, 0x00,
	0x00, 0x66, 0x66, 0x66, 0x66, 0x7C, 0x60, 0xC0,
	0x00, 0x76, 0xDC, 0x18, 0x18, 0x18, 0x18, 0x00,
	0xFC, 0x30, 0x78, 0xCC, 0xCC, 0x78, 0x30, 0xFC,
	0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x6C, 0x38, 0x00,
	0x38, 0x6C, 0xC6, 0xC6, 0x6C, 0x6C, 0xEE, 0x00,
	0x1C, 0x30, 0x18, 0x7C, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0x7E, 0xDB, 0xDB, 0x7E, 0x00, 0x00,
	0x06, 0x0C, 0x7E, 0xDB, 0xDB, 0x7E, 0x60, 0xC0,
	0x38, 0x60, 0xC0, 0xF8, 0xC0, 0x60, 0x38, 0x00,
	0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x00,
	0x00, 0xFC, 0x00, 0xFC, 0x00, 0xFC, 0x00, 0x00,
	0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0xFC, 0x00,
	0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xFC, 0x00,
	0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xFC, 0x00,
	0x0E, 0x1B, 0x1B, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0xD8, 0xD8, 0x70,
	0x30, 0x30, 0x00, 0xFC, 0x00, 0x30, 0x30, 0x00,
	0x00, 0x76, 0xDC, 0x00, 0x76, 0xDC, 0x00, 0x00,
	0x38, 0x6C, 0x6C, 0x38, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
	0x0F, 0x0C, 0x0C, 0x0C, 0xEC, 0x6C, 0x3C, 0x1C,
	0x78, 0x6C, 0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00,
	0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x3C, 0x3C, 0x3C, 0x3C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

// Extended set (0x80-0xFF) for Russian versions of games
static const uint8 fontData_ExtendedRussian[] = {
	0x1E, 0x36, 0x66, 0x66, 0x7E, 0x66, 0x66, 0x00, // 0x80
	0x7C, 0x60, 0x60, 0x7C, 0x66, 0x66, 0x7C, 0x00,
	0x7C, 0x66, 0x66, 0x7C, 0x66, 0x66, 0x7C, 0x00,
	0x7E, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x00,
	0x38, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0xFE, 0xC6,
	0x7E, 0x60, 0x60, 0x7C, 0x60, 0x60, 0x7E, 0x00,
	0xDB, 0xDB, 0x7E, 0x3C, 0x7E, 0xDB, 0xDB, 0x00,
	0x3C, 0x66, 0x06, 0x1C, 0x06, 0x66, 0x3C, 0x00,
	0x66, 0x66, 0x6E, 0x7E, 0x76, 0x66, 0x66, 0x00,
	0x3C, 0x66, 0x6E, 0x7E, 0x76, 0x66, 0x66, 0x00,
	0x66, 0x6C, 0x78, 0x70, 0x78, 0x6C, 0x66, 0x00,
	0x1E, 0x36, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
	0xC6, 0xEE, 0xFE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00,
	0x66, 0x66, 0x66, 0x7E, 0x66, 0x66, 0x66, 0x00,
	0x3C, 0x66, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0x7E, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
	0x7C, 0x66, 0x66, 0x66, 0x7C, 0x60, 0x60, 0x00,
	0x3C, 0x66, 0x60, 0x60, 0x60, 0x66, 0x3C, 0x00,
	0x7E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x66, 0x66, 0x66, 0x3E, 0x06, 0x66, 0x3C, 0x00,
	0x7E, 0xDB, 0xDB, 0xDB, 0x7E, 0x18, 0x18, 0x00,
	0x66, 0x66, 0x3C, 0x18, 0x3C, 0x66, 0x66, 0x00,
	0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7F, 0x03,
	0x66, 0x66, 0x66, 0x3E, 0x06, 0x06, 0x06, 0x00,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xFF, 0x00,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xFF, 0x03,
	0xE0, 0x60, 0x60, 0x7C, 0x66, 0x66, 0x7C, 0x00,
	0xC6, 0xC6, 0xC6, 0xF6, 0xDE, 0xDE, 0xF6, 0x00,
	0x60, 0x60, 0x60, 0x7C, 0x66, 0x66, 0x7C, 0x00,
	0x78, 0x8C, 0x06, 0x3E, 0x06, 0x8C, 0x78, 0x00,
	0xCE, 0xDB, 0xDB, 0xFB, 0xDB, 0xDB, 0xCE, 0x00,
	0x3E, 0x66, 0x66, 0x66, 0x3E, 0x36, 0x66, 0x00,
	0x00, 0x00, 0x3C, 0x06, 0x3E, 0x66, 0x3A, 0x00,
	0x00, 0x3C, 0x60, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x00, 0x00, 0x7C, 0x66, 0x7C, 0x66, 0x7C, 0x00,
	0x00, 0x00, 0x7E, 0x60, 0x60, 0x60, 0x60, 0x00,
	0x00, 0x00, 0x3C, 0x6C, 0x6C, 0x6C, 0xFE, 0xC6,
	0x00, 0x00, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0x00, 0x00, 0xDB, 0x7E, 0x3C, 0x7E, 0xDB, 0x00,
	0x00, 0x00, 0x3C, 0x66, 0x0C, 0x66, 0x3C, 0x00,
	0x00, 0x00, 0x66, 0x6E, 0x7E, 0x76, 0x66, 0x00,
	0x00, 0x18, 0x66, 0x6E, 0x7E, 0x76, 0x66, 0x00,
	0x00, 0x00, 0x66, 0x6C, 0x78, 0x6C, 0x66, 0x00,
	0x00, 0x00, 0x1E, 0x36, 0x66, 0x66, 0x66, 0x00,
	0x00, 0x00, 0xC6, 0xFE, 0xFE, 0xD6, 0xC6, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x7E, 0x66, 0x66, 0x00,
	0x00, 0x00, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0x00, 0x00, 0x7E, 0x66, 0x66, 0x66, 0x66, 0x00,
	0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,
	0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA,
	0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77, 0xDD, 0x77,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0xF8, 0x18, 0x18, 0x18, 0x18,
	0x18, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18, 0x18,
	0x36, 0x36, 0x36, 0xF6, 0x36, 0x36, 0x36, 0x36,
	0x00, 0x00, 0x00, 0xFE, 0x36, 0x36, 0x36, 0x36,
	0x00, 0xF8, 0x18, 0xF8, 0x18, 0x18, 0x18, 0x18,
	0x36, 0xF6, 0x06, 0xF6, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x00, 0xFE, 0x06, 0xF6, 0x36, 0x36, 0x36, 0x36,
	0x36, 0xF6, 0x06, 0xFE, 0x00, 0x00, 0x00, 0x00,
	0x36, 0x36, 0x36, 0xFE, 0x00, 0x00, 0x00, 0x00,
	0x18, 0xF8, 0x18, 0xF8, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xF8, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x1F, 0x00, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x18, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xFF, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x1F, 0x18, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x18, 0xFF, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18, 0x18,
	0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x37, 0x30, 0x3F, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x3F, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36,
	0x36, 0xF7, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xFF, 0x00, 0xF7, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36,
	0x00, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x36, 0xF7, 0x00, 0xF7, 0x36, 0x36, 0x36, 0x36,
	0x18, 0xFF, 0x00, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x36, 0x36, 0x36, 0xFF, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xFF, 0x00, 0xFF, 0x18, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0xFF, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x3F, 0x00, 0x00, 0x00, 0x00,
	0x18, 0x1F, 0x18, 0x1F, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1F, 0x18, 0x1F, 0x18, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x3F, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0xFF, 0x36, 0x36, 0x36, 0x36,
	0x18, 0xFF, 0x18, 0xFF, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0xF8, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x1F, 0x18, 0x18, 0x18, 0x18,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x66, 0x66, 0x7C, 0x60, 0x00,
	0x00, 0x00, 0x3C, 0x66, 0x60, 0x66, 0x3C, 0x00,
	0x00, 0x00, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x3E, 0x06, 0x7C, 0x00,
	0x00, 0x00, 0x7E, 0xDB, 0xDB, 0x7E, 0x18, 0x00,
	0x00, 0x00, 0x66, 0x3C, 0x18, 0x3C, 0x66, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x7F, 0x03,
	0x00, 0x00, 0x66, 0x66, 0x3E, 0x06, 0x06, 0x00,
	0x00, 0x00, 0xDB, 0xDB, 0xDB, 0xDB, 0xFF, 0x00,
	0x00, 0x00, 0xDB, 0xDB, 0xDB, 0xDB, 0xFF, 0x03,
	0x00, 0x00, 0xE0, 0x60, 0x7C, 0x66, 0x7C, 0x00,
	0x00, 0x00, 0xC6, 0xC6, 0xF6, 0xDE, 0xF6, 0x00,
	0x00, 0x00, 0x60, 0x60, 0x7C, 0x66, 0x7C, 0x00,
	0x00, 0x00, 0x7C, 0x06, 0x3E, 0x06, 0x7C, 0x00,
	0x00, 0x00, 0xCE, 0xDB, 0xFB, 0xDB, 0xCE, 0x00,
	0x00, 0x00, 0x3E, 0x66, 0x3E, 0x36, 0x66, 0x00,
	0x00, 0x00, 0xFE, 0x00, 0xFE, 0x00, 0xFE, 0x00,
	0x10, 0x10, 0x7C, 0x10, 0x10, 0x00, 0x7C, 0x00,
	0x00, 0x30, 0x18, 0x0C, 0x06, 0x0C, 0x18, 0x30,
	0x00, 0x0C, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0C,
	0x0E, 0x1B, 0x1B, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x18, 0xD8, 0xD8, 0x70,
	0x00, 0x18, 0x18, 0x00, 0x7E, 0x00, 0x18, 0x18,
	0x00, 0x76, 0xDC, 0x00, 0x76, 0xDC, 0x00, 0x00,
	0x00, 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00,
	0x03, 0x02, 0x06, 0x04, 0xCC, 0x68, 0x38, 0x10,
	0x3C, 0x42, 0x99, 0xA1, 0xA1, 0x99, 0x42, 0x3C,
	0x30, 0x48, 0x10, 0x20, 0x78, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x7C, 0x7C, 0x7C, 0x7C, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x7E, 0x00
};

void GfxFont::init() {
	if (ConfMan.getBool("herculesfont")) {
		// User wants, that we use Hercules hires font, try to load it
		loadFontHercules();
	} else {
		switch (_vm->_renderMode) {
		case Common::kRenderHercA:
		case Common::kRenderHercG:
			// Render mode is Hercules, we try to load Hercules hires font
			loadFontHercules();
			break;
		default:
			break;
		}
	}

	if (!_fontData) {
		switch (_vm->_renderMode) {
		case Common::kRenderAmiga:
			// Try user-file first, if that fails use our internal inaccurate topaz font
			loadFontScummVMFile("agi-font-amiga.bin");
			if (!_fontData) {
				loadFontAmigaPseudoTopaz();
			}
			break;
		case Common::kRenderApple2GS:
			// Special font, stored in file AGIFONT
			loadFontAppleIIgs();
			break;
		case Common::kRenderAtariST:
			// TODO: Atari ST uses another font
			// Seems to be the standard Atari ST 8x8 system font
			loadFontScummVMFile("agi-font-atarist.bin");
			if (!_fontData) {
				loadFontAtariST("agi-font-atarist-system.fnt");
				if (!_fontData) {
					// TODO: in case we find a recreation of the font, add it in here
				}
			}
			break;
		case Common::kRenderHercA:
		case Common::kRenderHercG:
		case Common::kRenderCGA:
		case Common::kRenderEGA:
		case Common::kRenderVGA:
			switch (_vm->getGameID()) {
			case GID_MICKEY:
				// load mickey mouse font from interpreter file
				loadFontMickey();
				break;
			default:
				loadFontScummVMFile("agi-font-dos.bin");
				break;
			}
			break;

		default:
			break;
		}

		if (!_fontData) {
			// no font assigned?
			// use regular PC-BIOS font (taken from Dos-Box with a few modifications)
			_fontData = fontData_PCBIOS;
			debug("AGI: Using PC-BIOS font");
		}
	}

	if (_vm->getLanguage() == Common::RU_RUS) {
		// Russian versions need special extended set
		overwriteExtendedWithRussianSet();
	}
}

const byte *GfxFont::getFontData() {
	assert(_fontData);
	return _fontData;
}

bool GfxFont::isFontHires() {
	return _fontIsHires;
}

void GfxFont::overwriteSaveRestoreDialogCharacter() {
	// overwrite character 0x1A with the standard Sierra arrow to the right character
	// required for the original save/restore dialogs
	memcpy(_fontDataAllocated + (0x1A * 8), fontData_ArrowRightCharacter, sizeof(fontData_ArrowRightCharacter));
}

// Overwrite extended character set (0x80-0xFF) with Russian characters
void GfxFont::overwriteExtendedWithRussianSet() {
	if (_fontIsHires) {
		// TODO: Implement overwriting hires font characters too
		return;
	}

	if (!_fontDataAllocated) {
		// nothing allocated, we need to allocate space ourselves to be able to modify an internal font
		_fontDataAllocated = (uint8 *)calloc(256, 8);
		memcpy(_fontDataAllocated, _fontData, 128 * 8); // copy ASCII set over
		_fontData = _fontDataAllocated;
	}
	// Overwrite extended set with Russian characters
	memcpy(_fontDataAllocated + (128 * 8), fontData_ExtendedRussian, 128 * 8);

	debug("AGI: Using Russian extended font set");
}

// This code loads a ScummVM-specific user-supplied binary font file
// It's assumed that it's a plain binary file, that contains 256 characters. 8 bytes per character.
// 8x8 pixels per character. File size 2048 bytes.
//
// Currently used for:
//  Atari ST - "agi-font-atarist.bin" -> should be the Atari ST 8x8 system font
//  Amiga    - "agi-font-amiga.bin"   -> should be the Amiga 8x8 Topaz font
//  DOS      - "agi-font-dos.bin"
void GfxFont::loadFontScummVMFile(Common::String fontFilename) {
	Common::File fontFile;
	int32 fontFileSize = 0;

	if (!fontFile.open(fontFilename)) {
		// Continue, if file not found
		// These ScummVM font files are totally optional, so don't show a warning
		return;
	}

	fontFileSize = fontFile.size();
	if (fontFileSize != (256 * 8)) {
		// unexpected file size
		fontFile.close();
		warning("Fontfile '%s': unexpected file size", fontFilename.c_str());
		return;
	}

	// allocate space for font bitmap data
	_fontDataAllocated = (uint8 *)calloc(256, 8);
	_fontData = _fontDataAllocated;

	// read font data, is already in the format that we need (plain bitmap 8x8)
	fontFile.read(_fontDataAllocated, 256 * 8);
	fontFile.close();

	overwriteSaveRestoreDialogCharacter();

	debug("AGI: Using user-supplied font");
}

// We load the Mickey Mouse font from MICKEY.EXE
void GfxFont::loadFontMickey() {
	Common::File interpreterFile;
	int32 interpreterFileSize = 0;
	byte *fontData = nullptr;

	if (!interpreterFile.open("mickey.exe")) {
		// Continue, if file not found
		warning("Could not open file 'mickey.exe' for Mickey Mouse font");
		return;
	}

	interpreterFileSize = interpreterFile.size();
	if (interpreterFileSize != 55136) {
		// unexpected file size
		interpreterFile.close();
		warning("File 'mickey.exe': unexpected file size");
		return;
	}
	interpreterFile.seek(32476); // offset of font data

	// allocate space for font bitmap data
	fontData = (uint8 *)calloc(256, 8);
	_fontData = fontData;
	_fontDataAllocated = fontData;

	// read font data, is already in the format that we need (plain bitmap 8x8)
	interpreterFile.read(fontData, 256 * 8);
	interpreterFile.close();

	debug("AGI: Using Mickey Mouse font");
}

// we create a bitmap out of the topaz data used in parallaction (which is normally found in staticres.cpp)
// it's a recreation of the Amiga Topaz font but not really accurate
void GfxFont::loadFontAmigaPseudoTopaz() {
	Graphics::Surface surf;
	surf.create(8, 8, Graphics::PixelFormat::createFormatCLUT8());
	byte *fontData = (uint8 *)calloc(256, 8);

	_fontData = fontData;
	_fontDataAllocated = fontData;

	// copy first 32 PC-BIOS characters over
	memcpy(fontData, fontData_PCBIOS, FONT_DISPLAY_WIDTH * 32);
	fontData += FONT_DISPLAY_WIDTH * 32;

	Graphics::AmigaFont topaz;

	for (uint16 curChar = topaz.getLoChar(); curChar <= topaz.getHiChar(); curChar++) {
		topaz.drawChar(&surf, curChar, 0, 0, 0xff);

		// Turn it into 1bpp
		for (int y = 0; y < 8; y++) {
			byte line = 0;
			byte *ptr = (byte *)surf.getBasePtr(0, y);
			for (int x = 0; x < 8; x++, ptr++) {
				line <<= 1;
				line |= *ptr & 0x1;
			}
			*fontData++ = line;
		}
		memset(surf.getPixels(), 0, 8 * 8);
	}

	surf.free();

	debug("AGI: Using recreation of Amiga Topaz font");
}

void GfxFont::loadFontAppleIIgs() {
	Common::File fontFile;
	uint16 headerIIgs_OffsetMacHeader = 0;
	uint16 headerIIgs_Version = 0;
	uint16 macRecord_FirstChar = 0;
	uint16 macRecord_LastChar = 0;
	int16 macRecord_MaxKern = 0;
	uint16 macRecord_RectHeight = 0;
	uint16 macRecord_StrikeWidth = 0;
	uint16 strikeDataLen = 0;
	byte *strikeDataPtr = nullptr;
	uint16 actualCharacterCount = 0;
	uint16 totalCharacterCount = 0;
	uint16 *locationTablePtr = nullptr;
	uint16 *offsetWidthTablePtr = nullptr;

	uint16 curCharNr = 0;
	uint16 curRow = 0;
	uint16 curLocation = 0;
	uint16 curLocationBytes = 0;
	uint16 curLocationBits = 0;
	uint16 curCharOffsetWidth = 0;
	uint16 curCharOffset = 0;
	uint16 curCharWidth = 0;
	uint16 curStrikeWidth = 0;

	uint16 curPixelNr = 0;
	uint16 curBitMask = 0;
	int16 positionAdjust = 0;
	byte curByte = 0;
	byte fontByte = 0;

	uint16 strikeRowOffset = 0;
	uint16 strikeCurOffset = 0;

	byte *fontData = nullptr;

	if (!fontFile.open("agifont")) {
		// Continue,
		// This also happens when the user selected Apple IIgs as render for the palette for non-AppleIIgs games
		warning("Could not open file 'agifont' for Apple IIgs font");
		return;
	}

	// Apple IIgs header
	headerIIgs_OffsetMacHeader = fontFile.readUint16LE();
	fontFile.skip(2); // font family
	fontFile.skip(2); // font style
	fontFile.skip(2); // point size
	headerIIgs_Version = fontFile.readUint16LE();
	fontFile.skip(2); // bounds type
	// end of Apple IIgs header
	// Macintosh font record
	fontFile.skip(2); // font type
	macRecord_FirstChar = fontFile.readUint16LE();
	macRecord_LastChar = fontFile.readUint16LE();
	fontFile.skip(2); // max width
	macRecord_MaxKern = fontFile.readSint16LE();
	fontFile.skip(2); // negative descent
	fontFile.skip(2); // rect width
	macRecord_RectHeight = fontFile.readUint16LE();
	fontFile.skip(2); // low word ptr table
	fontFile.skip(2); // font ascent
	fontFile.skip(2); // font descent
	fontFile.skip(2); // leading
	macRecord_StrikeWidth = fontFile.readUint16LE();

	// security-checks
	if (headerIIgs_OffsetMacHeader != 6)
		error("AppleIIgs-font: unexpected header");
	if (headerIIgs_Version != 0x0101)
		error("AppleIIgs-font: not a 1.1 font");
	if ((macRecord_FirstChar != 0) || (macRecord_LastChar != 255))
		error("AppleIIgs-font: unexpected characters");
	if (macRecord_RectHeight != 8)
		error("AppleIIgs-font: expected 8x8 font");

	// Calculate table sizes
	strikeDataLen = macRecord_StrikeWidth * macRecord_RectHeight * 2;
	actualCharacterCount = (macRecord_LastChar - macRecord_FirstChar + 1);
	totalCharacterCount = actualCharacterCount + 2; // replacement-char + extra character

	// Allocate memory for tables
	strikeDataPtr = (byte *)calloc(strikeDataLen, 1);
	locationTablePtr = (uint16 *)calloc(totalCharacterCount, 2); // 1 word per character
	offsetWidthTablePtr = (uint16 *)calloc(totalCharacterCount, 2); // ditto

	// read tables
	fontFile.read(strikeDataPtr, strikeDataLen);
	for (curCharNr = 0; curCharNr < totalCharacterCount; curCharNr++) {
		locationTablePtr[curCharNr] = fontFile.readUint16LE();
	}
	for (curCharNr = 0; curCharNr < totalCharacterCount; curCharNr++) {
		offsetWidthTablePtr[curCharNr] = fontFile.readUint16LE();
	}
	fontFile.close();

	// allocate space for font bitmap data
	fontData = (uint8 *)calloc(256, 8);
	_fontData = fontData;
	_fontDataAllocated = fontData;

	// extract font bitmap data
	for (curCharNr = 0; curCharNr < actualCharacterCount; curCharNr++) {
		curCharOffsetWidth = offsetWidthTablePtr[curCharNr];
		curLocation = locationTablePtr[curCharNr];
		if (curCharOffsetWidth == 0xFFFF) {
			// character does not exist in font, use replacement character instead
			curCharOffsetWidth = offsetWidthTablePtr[actualCharacterCount];
			curLocation = locationTablePtr[actualCharacterCount];
			curStrikeWidth = locationTablePtr[actualCharacterCount + 1] - curLocation;
		} else {
			curStrikeWidth = locationTablePtr[curCharNr + 1] - curLocation;
		}

		// Figure out bytes + bits location
		curLocationBytes = curLocation >> 3;
		curLocationBits = curLocation & 0x0007;
		curCharWidth = curCharOffsetWidth & 0x00FF; // isolate width
		curCharOffset = curCharOffsetWidth >> 8; // isolate offset

		if (!curCharWidth) {
			fontData += 8; // skip over this character
			continue;
		}

		if (curCharWidth != 8) {
			if (curCharNr != 0x3B)
				error("AppleIIgs-font: expected 8x8 font");
		}

		// Get all rows of the current character
		strikeRowOffset = 0;
		for (curRow = 0; curRow < macRecord_RectHeight; curRow++) {
			strikeCurOffset = strikeRowOffset + curLocationBytes;

			// Copy over bits
			fontByte = 0;
			curByte = strikeDataPtr[strikeCurOffset];
			curBitMask = 0x80 >> curLocationBits;

			for (curPixelNr = 0; curPixelNr < curStrikeWidth; curPixelNr++) {
				fontByte = fontByte << 1;
				if (curByte & curBitMask) {
					fontByte |= 0x01;
				}
				curBitMask = curBitMask >> 1;
				if (!curBitMask) {
					curByte = strikeDataPtr[strikeCurOffset + 1];
					curBitMask = 0x80;
				}
			}

			// adjust, so that it's aligned to the left (starting at 0x80 bit)
			fontByte = fontByte << (8 - curStrikeWidth);

			// now adjust according to offset + MaxKern
			positionAdjust = macRecord_MaxKern + curCharOffset;

			// adjust may be negative for space, or 8 for "empty" characters
			if (positionAdjust > 8)
				error("AppleIIgs-font: invalid character spacing");

			if (positionAdjust < 0) {
				// negative adjust strangely happens for empty characters like space
				if (curStrikeWidth)
					error("AppleIIgs-font: invalid character spacing");
			}

			if (positionAdjust > 0) {
				// move the amount of pixels to the right
				fontByte = fontByte >> positionAdjust;
			}

			*fontData = fontByte;
			fontData++;

			strikeRowOffset += macRecord_StrikeWidth * 2;
		}
	}

	free(offsetWidthTablePtr);
	free(locationTablePtr);
	free(strikeDataPtr);

	overwriteSaveRestoreDialogCharacter();

	debug("AGI: Using Apple IIgs font");
}

// Loads Atari ST font file
// It's found inside Atari ST ROMs. Just search for "8x8 system font". Font starts 4 bytes before that.
void GfxFont::loadFontAtariST(Common::String fontFilename) {
	Common::File fontFile;
	uint16 header_FirstChar = 0;
	uint16 header_LastChar = 0;
	uint16 header_MaxWidth = 0;
	uint16 header_MaxHeight = 0;
	uint16 header_Flags = 0;
	//uint32 header_OffsetOfCharOffsets = 0;
	//uint32 header_OffsetOfFontData = 0;
	uint16 header_FormWidth = 0;
	uint16 header_FormHeight = 0;
	uint16 totalCharacterCount = 0;
	uint16 *charOffsetTablePtr = nullptr;
	byte *rawDataTablePtr = nullptr;

	uint16 curCharNr = 0;
	uint16 curCharRawOffset = 0;
	uint16 curCharDestOffset = 0;
	uint16 curRow = 0;

	byte *fontData = nullptr;

	if (!fontFile.open(fontFilename)) {
		// Continue, if file not found
		warning("Could not open file 'agi-font-atarist-system.bin' for Atari ST 8x8 system font");
		return;
	}

	// Atari ST font header
	fontFile.skip(2); // face identifier
	fontFile.skip(2); // point size
	fontFile.skip(32); // font name
	header_FirstChar = fontFile.readUint16BE();
	header_LastChar = fontFile.readUint16BE();
	fontFile.skip(10); // aligntment of cells
	header_MaxWidth = fontFile.readUint16BE();
	header_MaxHeight = fontFile.readUint16BE();
	fontFile.skip(2); // left offset cel
	fontFile.skip(2); // right offset cel
	fontFile.skip(2); // number of pixels to thicken pixels
	fontFile.skip(2); // underline width
	fontFile.skip(2); // lightning mask
	fontFile.skip(2); // skewing mask
	header_Flags = fontFile.readUint16BE();
	// bit 0 - default system font
	// bit 1 - horizontal offset table (not supported)
	// bit 2 - byte orientation word is high->low
	// bit 3 - mono spaced font
	fontFile.skip(4); // horizontal table offset
	fontFile.skip(4); // header_OffsetOfCharOffsets = fontFile.readUint32BE();
	fontFile.skip(4); // header_OffsetOfFontData = fontFile.readUint32BE();
	header_FormWidth = fontFile.readUint16BE();
	header_FormHeight = fontFile.readUint16BE();
	fontFile.skip(4); // pointer to next font

	totalCharacterCount = header_LastChar - header_FirstChar + 1;

	// security-checks
	if (header_MaxWidth > 8)
		error("AtariST-font: not a 8x8 font");
	if (header_MaxHeight != 8)
		error("AtariST-font: not a 8x8 font");
	if (header_FormHeight != 8)
		error("AtariST-font: not a 8x8 font");
	if ((header_FirstChar != 0) || (header_LastChar != 255))
		error("AtariST-font: unexpected characters");
	if (header_FormWidth != totalCharacterCount)
		error("AtariST-font: header inconsistency");
	if (!(header_Flags & 0x04))
		error("AtariST-font: font data not in high->low order");
	if (!(header_Flags & 0x08))
		error("AtariST-font: not a mono-spaced font");

	// Now we should normally use the offsets, but they don't make sense to me
	// So I just read the data directly. For the 8x8 system font that works
	fontFile.skip(2); // extra bytes

	// Allocate memory for tables
	charOffsetTablePtr = (uint16 *)calloc(totalCharacterCount, 2); // 1 word per character
	rawDataTablePtr = (byte *)calloc(header_FormWidth, header_FormHeight);

	// Char-Offset Table (2 * total number of characters)
	for (curCharNr = 0; curCharNr < totalCharacterCount; curCharNr++) {
		charOffsetTablePtr[curCharNr] = fontFile.readUint16BE();
	}

	// Followed by actual font data
	// Attention: Atari ST fonts contain every same row of all characters after each other.
	// So it's basically like this:
	// [character data of first row of first character]
	// [character data of first row of second character]
	// ...
	// [character data of first row of last character]
	// [character data of second row of first character]
	fontFile.skip(2); // extra bytes
	fontFile.read(rawDataTablePtr, header_FormWidth * header_FormHeight);
	fontFile.close();

	// allocate space for font bitmap data
	fontData = (uint8 *)calloc(256, 8);
	_fontData = fontData;
	_fontDataAllocated = fontData;

	// extract font bitmap data
	for (curCharNr = 0; curCharNr < totalCharacterCount; curCharNr++) {
		// Figure out base offset from char offset table
		curCharRawOffset = charOffsetTablePtr[curCharNr] >> 3;
		curCharDestOffset = curCharNr * 8; // destination offset into our font data

		// now copy over every row of the character
		for (curRow = 0; curRow < header_FormHeight; curRow++) {
			fontData[curCharDestOffset] = rawDataTablePtr[curCharRawOffset];
			curCharDestOffset++;
			curCharRawOffset += header_FormWidth;
		}
	}

	free(rawDataTablePtr);
	free(charOffsetTablePtr);

	overwriteSaveRestoreDialogCharacter();

	debug("AGI: Using Atari ST 8x8 system font");
}

// Loads a Sierra Hercules font file
void GfxFont::loadFontHercules() {

	if (_vm->getLanguage() == Common::RU_RUS) {
		warning("Hercules font does not contain Russian characters, switching to default");

		return;
	}

	Common::File fontFile;
	int32 fontFileSize = 0;
	byte *fontData = nullptr;
	byte *rawData = nullptr;

	uint16 rawDataPos = 0;
	uint16 curCharNr = 0;
	uint16 curCharLine = 0;

	if (fontFile.open("hgc_font")) {
		// hgc_font file found, this is interleaved font data 16x12, should be 3072 bytes
		// 24 bytes per character, 128 characters
		fontFileSize = fontFile.size();
		if (fontFileSize == (128 * 24)) {
			// size seems to be fine
			fontData = (uint8 *)calloc(256, 32);
			_fontDataAllocated = fontData;

			rawData = (byte *)calloc(128, 24);
			fontFile.read(rawData, 128 * 24);

			// convert interleaved 16x12 -> non-interleaved 16x16
			for (curCharNr = 0; curCharNr < 128; curCharNr++) {
				fontData += 4; // skip the first 2 lines
				for (curCharLine = 0; curCharLine < 6; curCharLine++) {
					fontData[0] = rawData[rawDataPos + 2 + 0];
					fontData[1] = rawData[rawDataPos + 2 + 1];
					fontData[2] = rawData[rawDataPos + 0 + 0];
					fontData[3] = rawData[rawDataPos + 0 + 1];
					rawDataPos += 4;
					fontData += 4;
				}
				fontData += 4; // skip the last 2 lines
			}

			free(rawData);
		} else {
			warning("Fontfile 'hgc_font': unexpected file size");
		}
		fontFile.close();

	}

	// It seems hgc_graf.ovl holds a low-res font. It makes no real sense to use it.
	// This was only done to AGI3 games and those rendered differently (2 pixel lines -> 3 pixel lines instead of 4)
	// User could copy hgc_font from another AGI game over to get the hires font working.
#if 0
	if (!_fontDataAllocated) {
		if (fontFile.open("hgc_graf.ovl")) {
			// hgc_graf.ovl file found, this is font data + code. non-interleaved font data, should be 3075 bytes
			// 16 bytes per character, 128 characters, 2048 bytes of font data, starting offset 21
			fontFileSize = fontFile.size();
			if (fontFileSize == 3075) {
				// size seems to be fine
				fontData = (uint8 *)calloc(256, 32);
				_fontDataAllocated = fontData;

				fontFile.seek(21);
				rawData = (byte *)calloc(128, 16);
				fontFile.read(rawData, 128 * 16);

				// repeat every line 2 times to get 16x16 pixels
				for (curCharNr = 0; curCharNr < 128; curCharNr++) {
					for (curCharLine = 0; curCharLine < 8; curCharLine++) {
						fontData[0] = rawData[rawDataPos + 0];
						fontData[1] = rawData[rawDataPos + 1];
						fontData[2] = rawData[rawDataPos + 0];
						fontData[3] = rawData[rawDataPos + 1];
						rawDataPos += 2;
						fontData += 4;
					}
				}

				free(rawData);

			} else {
				warning("Fontfile 'hgc_graf.ovl': unexpected file size");
			}
			fontFile.close();
		}
	}
#endif

	if (_fontDataAllocated) {
		// font loaded
		_fontData = _fontDataAllocated;
		_fontIsHires = true;

		debug("AGI: Using Hercules hires font");

	} else {
		// Continue, if no file was not found
		warning("Could not open/use file 'hgc_font' for Hercules hires font");
		if (GUI::GuiManager::hasInstance()) {
			GUI::MessageDialog dialog(_("Could not open/use file 'hgc_font' for Hercules hires font.\nIf you have such file in other AGI (Sierra) game, you can copy it to the game directory"));
			dialog.runModal();
		};

	}
}

} // End of namespace Agi
